# GLM to support yield analysis
# Paired t-test as used by PD may not be fair comparison
# Instead use linear mixed effects model, with expt_# as random variable
# Author: Jim Stevens
# Date 19/3/24
# Version 4.3.3

#### Housekeeping ####
rm(list=ls())
library(readxl)
library(tidyverse)
library(car)
library(lme4)
library(cowplot)

#### Read in data ####

dataset <- read_excel("//cloud/project/Data/Harvest/Basil IAA Means of six experiments Jan 24 PAD.xlsx",
                      sheet="Long_data")
#View(dataset)
str(dataset)
dataset$Treatment <- as.factor(dataset$Treatment)
dataset$Treatment <- factor(dataset$Treatment, labels=c("Controlled","Feedback"))
dataset$Measurement <- as.factor(dataset$Measurement)

#### Check normality ####
# qqplots
norm.fw <- dataset %>%
  filter(Measurement=="fw")
qqPlot(norm.fw$Value) # looks ok
norm.la <- dataset %>%
  filter(Measurement=="area")
qqPlot(norm.la$Value) # looks ok
norm.dw <- dataset %>%
  filter(Measurement=="dw")
qqPlot(norm.la$Value) # looks ok
norm.kw <- dataset %>%
  filter(Measurement=="kW")

shapiro.test(norm.fw$Value)
# fail
shapiro.test(norm.la$Value)
# pass
shapiro.test(norm.dw$Value)
# fail
shapiro.test(norm.kw$Value)
# pass


#### Box Cox transformation of FW data ####

# boxcox fresh weight
bc.mod <- lm(Value~c(1:length(norm.fw$Value)),data=norm.fw) 
bc <- MASS::boxcox(norm.fw$Value~c(1:length(norm.fw$Value)))
lambda <- bc$x[which.max(bc$y)]

bc.new <- lm(((norm.fw$Value^lambda-1)/lambda) ~ c(1:length(norm.fw$Value)))

#define plotting area
op <- par(pty = "s", mfrow = c(1, 2))

#Q-Q plot for original model
qqnorm(bc.mod$residuals)
qqline(bc.mod$residuals)

#Q-Q plot for Box-Cox transformed model
qqnorm(bc.new$residuals)
qqline(bc.new$residuals)

#display both Q-Q plots
par(op)

# The new model is frankly not that brilliant.
# We will definitely have to resort to an LME

#### Linear Mixed Effect Model ####

## LME vs LM for FW
# At biological replicate level (a bit cheeky, but to see if it works)
lme.fw <- lmer(norm.fw$Value~norm.fw$Treatment+(1|norm.fw$Expt_no))
summary(lme.fw)
Anova(lme.fw, type=3) # treatment is significant for fw

## Try lme at expt level not replicate level
sum.fw <- norm.fw %>%
  group_by(Expt_no,Treatment) %>%
  summarise(fw=mean(Value),.groups="drop")
lme.sum.fw <- lmer(sum.fw$fw~sum.fw$Treatment+(1|as.factor(sum.fw$Expt_no)))
summary(lme.sum.fw)
Anova(lme.sum.fw,type="III") # significant

capture.output(summary(lme.sum.fw),
               file="./Outputs/Final Plots/Stats/FW_model.txt")

lm.sum.fw <- lm(sum.fw$fw~sum.fw$Treatment)
summary(lm.sum.fw)
Anova(lm.sum.fw, type = 3)

anova(lme.sum.fw,lm.sum.fw)
# lme model has lower AIC => better fit.

## create df of predicted and residual values for plotting &c
op <- data.frame(predicted=predict(lme.sum.fw), 
                 residual = residuals(lme.sum.fw),
                 Expt=sum.fw$Expt_no)
# Plot residuals vs predicted values
ggplot(op,aes(x=predicted,y=residual,colour=as.factor(Expt))) + 
  geom_point() + 
  geom_hline(yintercept=0, lty=3)

# histogram of residuals
ggplot(op,aes(x=residual)) + 
  geom_histogram(bins=6, color="black")

# qq plot of residuals
ggplot(op,aes(sample=residual))+ 
  stat_qq()+
  stat_qq_line()

## LME vs LM for leaf area

lme.la <- lmer(norm.la$Value~norm.la$Treatment+(1|norm.la$Expt_no))
summary(lme.la)
Anova(lme.la,type=3) # treatment is significant for la

## Try lme at expt level not replicate level
sum.la <- norm.la %>%
  group_by(Expt_no,Treatment) %>%
  summarise(la=mean(Value),.groups="keep")
lme.sum.la <- lmer(sum.la$la~sum.la$Treatment+
                     (1|as.factor(sum.la$Expt_no)))
summary(lme.sum.la)
Anova(lme.sum.la,type=3)
capture.output(summary(lme.sum.la),
               file="./Outputs/Final Plots/Stats/LA_model.txt")

lm.sum.la <- lm(sum.la$la~sum.la$Treatment)
summary(lm.sum.la)
Anova(lm.sum.la, type = 3)
# No effect of treatment

anova(lme.sum.la,lm.sum.la)
# lme model has lower AIC => better fit.

## create df of predicted and residual values for plotting &c
op <- data.frame(predicted=predict(lme.sum.la), 
                 residual = residuals(lme.sum.la),
                 Expt=sum.la$Expt_no)
# Plot residuals vs predicted values
ggplot(op,aes(x=predicted,y=residual,colour=as.factor(Expt))) + 
  geom_point() + 
  geom_hline(yintercept=0, lty=3)

# histogram of residuals
ggplot(op,aes(x=residual)) + 
  geom_histogram(bins=6, color="black")

# qq plot of residuals
ggplot(op,aes(sample=residual))+ 
  stat_qq()+
  stat_qq_line()

## LME vs LM for dry weight

## Try lme at expt level not replicate level
sum.dw <- norm.dw %>%
  group_by(Expt_no,Treatment) %>%
  summarise(dw=mean(Value),.groups="drop")
lme.sum.dw <- lmer(sum.dw$dw~sum.dw$Treatment+
                     (1|as.factor(sum.dw$Expt_no)))
summary(lme.sum.dw)
Anova(lme.sum.dw,type=3)
capture.output(summary(lme.sum.dw),
               file="./Outputs/Final Plots/Stats/DW_model.txt")

lm.sum.dw <- lm(sum.dw$dw~sum.dw$Treatment)
summary(lm.sum.dw)
Anova(lm.sum.dw, type = 3)
# No effect of treatment

anova(lme.sum.dw,lm.sum.dw)
# lme model has lower AIC => better fit.

## create df of predicted and residual values for plotting &c
op <- data.frame(predicted=predict(lme.sum.dw), 
                 residual = residuals(lme.sum.dw),
                 Expt=sum.dw$Expt_no)
# Plot residuals vs predicted values
ggplot(op,aes(x=predicted,y=residual,colour=as.factor(Expt))) + 
  geom_point() + 
  geom_hline(yintercept=0, lty=3)

# histogram of residuals
ggplot(op,aes(x=residual)) + 
  geom_histogram(bins=6, color="black")

# qq plot of residuals
ggplot(op,aes(sample=residual))+ 
  stat_qq()+
  stat_qq_line()

## LME vs LM for power use

lme.kw <- lmer(norm.kw$Value~norm.kw$Treatment+(1|as.factor(norm.kw$Expt_no)))
summary(lme.kw)
Anova(lme.kw,type=3) # treatment is significant for power use
capture.output(summary(lme.kw),
               file="./Outputs/Final Plots/Stats/KW_model.txt")

## Add calculation of CO2e to KW_model above ##
# Get FW and correct from tray-1 to m-2 
co2calc1 <- sum.fw %>%
  group_by(Treatment) %>%
  summarise(Mean_fw=mean(fw))
co2calc2 <- norm.kw %>%
  group_by(Treatment) %>%
  summarise(Mean_kwh=mean(Value))

capture.output(cat(
cat("FW per m^-2 Control = ",co2calc1$Mean_fw[1],
          " FW g plant^-1 \n x 12 plants tray ^-1 / 0.09 m2 tray-1 = ",
          12*co2calc1$Mean_fw[1]/0.09," g FW m^-2 \n"),
cat("FW per m^-2 Feedback = ",co2calc1$Mean_fw[2],
          " g plant^-1 \n x 12 plants tray ^-1 / 0.09 m2 tray-1 = ",
          12*co2calc1$Mean_fw[2]/0.09," g FW m^-2 \n"),
cat("CO2e UK Grid (2023) = 207 g CO2e kWh^-1 \n"),
cat("Control kW h m-2 = ",co2calc2$Mean_kwh[1],
          " x 207 g CO2e kWh^-1 = ",co2calc2$Mean_kwh[1]*207,"g CO2 m^-2 \n"),
cat("Feedback kW h m-2 = ",co2calc2$Mean_kwh[2],
          " x 207 g CO2e kWh^-1 = ",co2calc2$Mean_kwh[2]*207,"g CO2 m^-2 \n"),
cat("Control g CO2e g FW^-1 = ", co2calc2$Mean_kwh[1]*207, " / ",
    12*co2calc1$Mean_fw[1]/0.09," = ", 
    round((co2calc2$Mean_kwh[1]*207)/(12*co2calc1$Mean_fw[1]/0.09),2),"\n"),
cat("Feedback g CO2e g FW^-1 = ", co2calc2$Mean_kwh[2]*207, " / ",
    12*co2calc1$Mean_fw[2]/0.09," = ", 
    round((co2calc2$Mean_kwh[2]*207)/(12*co2calc1$Mean_fw[2]/0.09),2),"\n"),
cat("Difference = ",
    round((co2calc2$Mean_kwh[1]*207)/(12*co2calc1$Mean_fw[1]/0.09),2)-round((co2calc2$Mean_kwh[2]*207)/(12*co2calc1$Mean_fw[2]/0.09),2) ,
          "g CO2e m^-2 \n"),
cat("Percent difference = ", 
    round(100*(((co2calc2$Mean_kwh[2]*207)/(12*co2calc1$Mean_fw[2]/0.09)) / ((co2calc2$Mean_kwh[1]*207)/(12*co2calc1$Mean_fw[1]/0.09)))-100,1) ,
    "%")),file="./Outputs/Final Plots/Stats/Emissions_Intensity_Calc.txt")

## Now create final .txt file for stats data in final plots folder ##

lm.kw <- lm(norm.kw$Value~norm.kw$Treatment)
summary(lm.kw)
Anova(lm.kw, type = 3)
# No effect of treatment

anova(lme.kw,lm.kw)
# Slightly better fit for LM vs LME, but nothing really in it.

## create df of predicted and residual values for plotting &c
op <- data.frame(predicted=predict(lme.kw), 
                 residual = residuals(lme.kw),
                 Expt=norm.kw$Expt_no)
# Plot residuals vs predicted values
ggplot(op,aes(x=predicted,y=residual,colour=as.factor(Expt))) + 
  geom_point() + 
  geom_hline(yintercept=0, lty=3)

# histogram of residuals
ggplot(op,aes(x=residual)) + 
  geom_histogram(bins=6, color="black")

# qq plot of residuals
ggplot(op,aes(sample=residual))+ 
  stat_qq()+
  stat_qq_line()

#### Derived data: Power use per unit o/p ####

# fresh mass per kW
der.fw <- left_join(sum.fw,norm.kw,by=c("Expt_no","Treatment")) %>%
  mutate(fw_per_kW=fw/Value)

lme.fw.kw <- lmer(der.fw$fw_per_kW~der.fw$Treatment+(1|as.factor(der.fw$Expt_no)))
summary(lme.fw.kw)
Anova(lme.fw.kw,type=3) 
# sig diff on energy efficiency per unit mass
capture.output(summary(lme.fw.kw),
               file="./Outputs/Final Plots/Stats/fw_per_kw_model.txt")

# Leaf area per kW
der.la<- left_join(sum.la,norm.kw,by=c("Expt_no","Treatment")) %>%
  mutate(la_per_kW=la/Value)
lme.la.kw <- lmer(der.la$la_per_kW~der.la$Treatment+(1|as.factor(der.la$Expt_no)))
summary(lme.la.kw)
Anova(lme.la.kw,type=3) 
# sig dif on energy efficiency per unit leaf area
capture.output(summary(lme.la.kw),
               file="./Outputs/Final Plots/Stats/la_per_kw_model.txt")

# Dry mass per kW
der.dw<- left_join(sum.dw,norm.kw,by=c("Expt_no","Treatment")) %>%
  mutate(dw_per_kW=dw/Value)
lme.dw.kw <- lmer(der.dw$dw_per_kW~der.dw$Treatment+(1|as.factor(der.dw$Expt_no)))
summary(lme.dw.kw)
Anova(lme.dw.kw,type=3) 
# sig dif on energy efficiency per unit dry mass
capture.output(summary(lme.dw.kw),
               file="./Outputs/Final Plots/Stats/dw_per_kw_model.txt")

# sla + rwc
sum.sla <- left_join(sum.fw,sum.dw,by=c("Expt_no","Treatment"))
sum.sla <- left_join(sum.sla,sum.la,by=c("Expt_no","Treatment"))
sum.sla$rwc <- (sum.sla$fw-sum.sla$dw)/sum.sla$fw
sum.sla$sla <- sum.sla$la/sum.sla$dw

# rwc
rwc <- sum.sla
lme.rwc <- lmer(rwc$rwc~rwc$Treatment+(1|as.factor(rwc$Expt_no)))
summary(lme.rwc)
Anova(lme.rwc,type=3) 
# treatment is not significant for relative water content. 
# The difference is not down to turgidity or something
capture.output(summary(lme.rwc),
               file="./Outputs/Final Plots/Stats/rwc_model.txt")

# sla

lme.sla <- lmer(sum.sla$sla~sum.sla$Treatment+(1|as.factor(sum.sla$Expt_no)))
summary(lme.sla)
Anova(lme.sla,type=3) 
# treatment is not significant for sla - effectively no difference in thickness
capture.output(summary(lme.sla),
               file="./Outputs/Final Plots/Stats/sla_model.txt")


#### Plots of outputs ####

# extract p values (and stars?)
pvals <- NULL
fwp <- Anova(lme.fw)$"Pr(>Chisq)"
lap <- Anova(lme.sum.la)$"Pr(>Chisq)"
dwp <- Anova(lme.sum.dw)$"Pr(>Chisq)"
kwp <- Anova(lme.kw)$"Pr(>Chisq)"
fwkw <- Anova(lme.fw.kw)$"Pr(>Chisq)"
lakw <- Anova(lme.la.kw)$"Pr(>Chisq)"
dwkw <- Anova(lme.dw.kw)$"Pr(>Chisq)"
rwcp <- Anova(lme.rwc)$"Pr(>Chisq)"
slap <- Anova(lme.sla)$"Pr(>Chisq)"
pvals <- as.data.frame(rbind(fwp,lap,dwp,kwp,fwkw,lakw,dwkw,rwcp,slap))
pvals <- pvals %>% rename_at('V1', ~'p')
pvals <- pvals %>%
  mutate(Star=case_when(
    p<=0.001 ~ "***",
    p<=0.01  ~ "**",
    p<=0.05 ~ "*",
    p>0.05 ~ ""
  ))

# fw
a <- ggplot(sum.fw,aes(x=Treatment,y=fw,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'))+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11))+
  ylab(expression(paste("Fresh mass ( ",~g,~plant^-1," )"))) +
  annotate("segment",x=1,xend=2,y=max(sum.fw$fw*1.1))+
  annotate("text",x=1.5,y=unique(max(sum.fw$fw*1.15)),label=pvals[1,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, 
               color="black",fill="white")
a
ggsave("./Outputs/fw.png",a)

# la
b <- ggplot(sum.la,aes(x=Treatment,y=la,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'))+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11))+
  ylab(expression(paste("Leaf area (",~cm^2, ~plant^-1," )"))) +
  annotate("segment",x=1,xend=2,y=max(sum.la$la*1.1))+
  annotate("text",x=1.5,y=unique(max(sum.la$la*1.15)),label=pvals[2,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
b
ggsave("./Outputs/la.png",b)

# dw
b1 <- ggplot(sum.dw,aes(x=Treatment,y=dw,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'))+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11)) +
  ylab(expression(paste("Dry mass (",~g, ~plant^-1," )"))) +
  annotate("segment",x=1,xend=2,y=max(sum.dw$dw*1.1))+
  annotate("text",x=1.5,y=unique(max(sum.dw$dw*1.15)),label=pvals[3,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
b1
ggsave("./Outputs/dw.png",b1)

# kWh
c <- ggplot(norm.kw,aes(x=Treatment,y=Value,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'),outlier.shape = NA)+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11))+
  ylab(expression(paste("Total energy use (",~Sigma, ~kW, ~h," )"))) +
  annotate("segment",x=1,xend=2,y=max(norm.kw$Value*1.1))+
  annotate("text",x=1.5,y=unique(max(norm.kw$Value*1.15)),label=pvals[4,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
c
ggsave("./Outputs/kw.png",c)

# fw / kWh

d <- ggplot(der.fw,aes(x=Treatment,y=fw_per_kW,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'),outlier.shape = NA)+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=11))+
  ylab(expression(paste("Fresh mass per unit power (",~g,~kWh^-1," )"))) +
  annotate("segment",x=1,xend=2,y=max(der.fw$fw_per_kW*1.1))+
  annotate("text",x=1.5,y=unique(max(der.fw$fw_per_kW*1.15)),label=pvals[5,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
d
ggsave("./Outputs/fw_per_kW.png",d)


# la / kWh

e <- ggplot(der.la,aes(x=Treatment,y=la_per_kW,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'))+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title = element_text(hjust = 0.5,size=11))+
  ylab(expression(paste("Leaf area per unit power (",~cm^2,~kWh^-1," )"))) +
  annotate("segment",x=1,xend=2,y=max(der.la$la_per_kW*1.1))+
  annotate("text",x=1.5,y=unique(max(der.la$la_per_kW*1.15)),label=pvals[6,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
e

ggsave("./Outputs/la_per_kW.png",e)

# dw / kWh

f <- ggplot(der.dw,aes(x=Treatment,y=dw_per_kW,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'))+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title = element_text(hjust = 0.5,size=11))+
  ylab(expression(paste("Dry mass per unit power (",~g," dw",~kWh^-1," )"))) +
  annotate("segment",x=1,xend=2,y=max(der.dw$dw_per_kW*1.1))+
  annotate("text",x=1.5,y=unique(max(der.dw$dw_per_kW*1.15)),label=pvals[7,2])+
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")

f
ggsave("./Outputs/dw_per_kW.png",f)

# rwc

g <- ggplot(sum.sla,aes(x=Treatment, y=rwc*100,colour=Treatment))+
              geom_boxplot(colour=c('darkgreen','goldenrod'),outlier.shape = NA)+
              geom_point(position="jitter",colour="red")+
              theme_bw()+
              theme(panel.grid=element_blank(),
                    axis.title = element_text(hjust = 0.5))+
              ylab("Relative water content ( % )") +
              stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
g
ggsave("./Outputs/RWC.png",a)
#ggsave("./Outputs/Final Plots/Supplementary Information/RWC.png",a)

# sla

h <- ggplot(sum.sla,aes(x=Treatment, y=sla,colour=Treatment))+
  geom_boxplot(colour=c('darkgreen','goldenrod'),outlier.shape = NA)+
  geom_point(position="jitter",colour="red")+
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title = element_text(hjust = 0.5))+
  ylab(expression(paste("Specific Leaf Area ( ", ~cm^2,~g^-1," )"))) +
  stat_summary(fun=mean, geom="point", shape=1, size=3, color="black")
h
ggsave("./Outputs/SLA.png",h)
#ggsave("./Outputs/Final Plots/Supplementary Information/SLA.png",h)

## Effect sizes

eff <- left_join(sum.sla,norm.kw,by=c("Expt_no","Treatment")) %>%
  dplyr::select(-c("Measurement","Rep","Date")) %>%
  rename("kw"="Value") %>%
  mutate(fw.kw=fw/kw,
         la.kw=la/kw,
         dw.kw=dw/kw) %>%
  group_by(Treatment) %>%  
  summarise(N=n(),mfw=mean(fw),
            mdw=mean(dw),
            mla=mean(la),
            mrwc=mean(rwc),
            msla=mean(sla),
            mkw=mean(kw),
            mfw.kw=mean(fw.kw),
            mla.kw=mean(la.kw),
            mdw.kw=mean(dw.kw))
          
eff.diff <- as.data.frame(cbind(type="Diff",
                  fw=diff(eff$mfw),
                  dw=diff(eff$mdw),
                  la=diff(eff$mla),
                  rwc=diff(eff$mrwc),
                  sla=diff(eff$msla),
                  kw=abs(diff(eff$mkw)),
                  fwkw=diff(eff$mfw.kw),
                  lakw=diff(eff$mla.kw),
                  dwkw=diff(eff$mdw.kw)))

eff.sd <- left_join(sum.sla,norm.kw,by=c("Expt_no","Treatment")) %>%
  dplyr::select(-c("Measurement","Rep","Date")) %>%
  rename("kw"="Value") %>%
  mutate(fw.kw=fw/kw,
         la.kw=la/kw,
         dw.kw=dw/kw) %>%
  summarise(fw=sd(fw),
            dw=sd(dw),
            la=sd(la),
            rwc=sd(rwc),
            sla=sd(sla),
            kw=sd(kw),
            fwkw=sd(fw.kw),
            lakw=sd(la.kw),
            dwkw=sd(dw.kw)) %>%
  mutate(type="SD",.before=fw)
eff.comb <- rbind(eff.diff,eff.sd)
eff.comb$fw <- as.numeric(eff.comb$fw)
eff.comb$dw <- as.numeric(eff.comb$dw)
eff.comb$la <- as.numeric(eff.comb$la)
eff.comb$rwc <- as.numeric(eff.comb$rwc)
eff.comb$sla <- as.numeric(eff.comb$sla)
eff.comb$kw <- as.numeric(eff.comb$kw)
eff.comb$fwkw <- as.numeric(eff.comb$fwkw)
eff.comb$lakw <- as.numeric(eff.comb$lakw)
eff.comb$dwkw <- as.numeric(eff.comb$dwkw)
eff.comb <- eff.comb %>%
  dplyr::select(-type)
eff1 <- eff.comb[1,]/eff.comb[2,]  
eff1 <- as.data.frame(t(eff1))

capture.output(eff1,file="./Outputs/Final Plots/Stats/Effect_Sizes.txt")

eff1 <- eff1 %>%
  rename(Size="1") %>%
  mutate(Meaning=case_when(
    Size<0.2 ~ "Negligible",
    Size<0.5 ~ "Small",
    Size<0.8 ~ "Medium",
    .default="Large"
  ))
eff1$Type <- as.factor(c("Fresh mass","Dry mass","Leaf area","RWC","SLA","Energy use",
               "FM / kWh","LA / kWh","DM / kWh"))
eff1$Type <- factor(eff1$Type,
                    levels=c("Fresh mass","Dry mass","Leaf area",
                                       "RWC","SLA","Energy use",
                             "FM / kWh","DM / kWh","LA / kWh"))


effplot <- ggplot(eff1,aes(x=Type,y=Size))+
  geom_rect(mapping=aes(xmin=0,xmax=11,ymin=0,ymax=0.2),fill="red") +
  geom_rect(mapping=aes(xmin=0,xmax=11,ymin=0.2,ymax=0.5),fill="goldenrod") +
  geom_rect(mapping=aes(xmin=0,xmax=11,ymin=0.5,ymax=0.8),fill="darkorange") +
  geom_rect(mapping=aes(xmin=0,xmax=11,ymin=0.8,ymax=1.4),fill="darkgreen") +
  geom_point(shape=2,stroke=0.6,size=1) +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.title=element_text(size=8),
        axis.text.x=element_text(size=5,angle=45,hjust=1),
        axis.text.y=element_text(size=5))+
  ylab("Effect Size (a.u.)") +
  xlab("Measurement") +
  scale_y_continuous(limits=c(0.0,1.4),
                     breaks=c(0,0.2,0.5,0.8,1.4),
                     expand=c(0,0))+
  scale_x_discrete(expand=c(0,1))+
  annotate("text",x=9.5,y=0.1,label="Negligible",hjust=0,size=2)+
  annotate("text",x=9.5,y=0.35,label="Small",hjust=0,size=2) +
  annotate("text",x=9.5,y=0.65,label="Medium",hjust=0,size=2) +
  annotate("text",x=9.5,y=1.1,label="Large",hjust=0,size=2) +
  geom_vline(xintercept=c(0.5,1.5,2.5,3.5,4.5,5.5,6.5,7.5,8.5),
             linetype="dashed",alpha=0.15)+
  geom_vline(xintercept=5.5,linetype="solid",alpha=0.5)
  
effplot
ggsave("./Outputs/Final Plots/Fig_9_Effect_size.png",effplot)

#### Combined Plots for Paper ####

## Figure 2: Primary results ##

fig2 <- plot_grid(a+theme(axis.title.x=element_blank(),
                          axis.text.x=element_blank(),
                          axis.title=element_text(size=11)),
                   b1+theme(axis.title.x=element_blank(),
                            axis.text.x=element_blank(),
                            axis.title=element_text(size=11)),
                   b+theme(axis.title.x=element_blank(),
                           axis.text.x=element_text(size=11),
                           axis.title=element_text(size=11)),
                   align="vh",
                  labels="AUTO",label_size=10,ncol=1)
fig2
fig2a <- plot_grid(fig2,NULL,ncol=2)
fig2a
save_plot("./Outputs/Final Plots/Fig_4_Main_Results.png",
          fig2,base_height=12,base_width=4)

fig3 <- plot_grid(c+theme(axis.title.x=element_blank(),
                          axis.text.x=element_blank(),
                          axis.title=element_text(size=11)),
                  d+theme(axis.title.x=element_blank(),
                          axis.text.x=element_blank(),
                          axis.text.y=element_text(size=11),
                          axis.title=element_text(size=11)),
                  f+theme(axis.title.x=element_blank(),
                          axis.text.y=element_text(size=11),
                          axis.title=element_text(size=11)),
                  e+theme(axis.title.x=element_blank(),
                          axis.text.y=element_text(size=11),
                          axis.title=element_text(size=11)),
                  align="vh", labels="AUTO",
                  label_size=8,ncol=2)
fig3
save_plot("./Outputs/Final Plots/Fig_5_Derived_data.png",
          fig3, base_width=8,base_height=8)

supp_harvest <- plot_grid(g,h)
supp_harvest
save_plot("./Outputs/Final Plots/Fig_6_RWC_SLA_plots.png",
          supp_harvest)

